'use strict';

/* --------------------------------------------------------------
 bulk_email_invoice.js 2019-09-18
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2019 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * Bulk Email Invoice Modal Controller
 *
 * This controller handles the orders bulk invoice emails modal where the user can send emails for the
 * selected orders. The controller is able to create missing invoices if needed.
 */
gx.controllers.module('bulk_email_invoice', ['modal', 'loading_spinner'], function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES
    // ------------------------------------------------------------------------

    /**
     * Module Selector
     *
     * @type {jQuery}
     */

    var $this = $(this);

    /**
     * Module Instance
     *
     * @type {Object}
     */
    var module = {
        bindings: {
            subject: $this.find('.subject')
        }
    };

    /**
     * Selector for the email list item.
     *
     * @type {String}
     */
    var emailListItemSelector = '.email-list-item';

    /**
     * Selector for the email list item ID.
     *
     * @type {String}
     */
    var emailListItemEmailSelector = '.email-input';

    /**
     * Selector for the latest invoice number of the list item
     *
     * @type {string}
     */
    var latestInvoiceIdSelector = '.latest-invoice-id';

    /**
     * Selector for the option that indicates if missing invoices should be created
     *
     * @type {String}
     */
    var createMissingInvoicesSelector = '.create-missing-invoices';

    /**
     * Selector for the modal content body layer.
     *
     * @type {String}
     */
    var modalContentSelector = '.modal-content';

    /**
     * GM PDF Order URL
     *
     * @type {String}
     */
    var gmPdfOrderUrl = jse.core.config.get('appUrl') + '/admin/gm_pdf_order.php';

    /**
     * Admin Request URL
     *
     * @type {String}
     */
    var adminRequestUrl = jse.core.config.get('appUrl') + '/admin/admin.php';

    /**
     * Loading Spinner Selector
     *
     * @type {jQuery|null}
     */
    var $spinner = null;

    // ------------------------------------------------------------------------
    // FUNCTIONS
    // ------------------------------------------------------------------------

    /**
     * Show/hide loading spinner.
     *
     * @param {Boolean} spinnerVisibility Whether to show or hide the spinner.
     */
    function _toggleSpinner(spinnerVisibility) {
        if (spinnerVisibility) {
            $spinner = jse.libs.loading_spinner.show($this.find(modalContentSelector), $this.css('z-index'));
        } else {
            jse.libs.loading_spinner.hide($spinner);
        }
    }

    /**
     * Parse subject and replace the placeholders with the variables.
     *
     * @param {Object} orderData Contains the order's data.
     * @param {Object} subjectData Contains the required subject's data.
     *
     * @return {String} Returns the final email subject string.
     */
    function _getParsedSubject(orderData, subjectData) {
        return module.bindings.subject.get().replace('{INVOICE_NUM}', subjectData.invoiceNumber).replace('{INVOICE_ID}', subjectData.invoiceNumber).replace('{INVOICE_DATE}', subjectData.invoiceDate).replace('{DATE}', subjectData.invoiceDate).replace('{ORDER_ID}', orderData.id);
    }

    /**
     * Handles the successful delivery of all messages.
     */
    function _handleDeliverySuccess() {
        var message = jse.core.lang.translate('BULK_MAIL_SUCCESS', 'gm_send_order');

        // Show success message in the admin info box.
        jse.libs.info_box.addSuccessMessage(message);

        $('.orders .table-main').DataTable().ajax.reload(null, false);
        $('.orders .table-main').orders_overview_filter('reload');

        // Hide modal and loading spinner.
        _toggleSpinner(false);
        $this.modal('hide');
    }

    /**
     * Handles the failure of the message delivery.
     */
    function _handleDeliveryFailure() {
        var title = jse.core.lang.translate('error', 'messages');
        var content = jse.core.lang.translate('BULK_MAIL_UNSUCCESS', 'gm_send_order');

        // Show error message in a modal.
        jse.libs.modal.message({ title: title, content: content });

        // Hide modal and the loading spinner and re-enable the send button.
        _toggleSpinner(false);
        $this.modal('hide');
    }

    /**
     * Get the IDs of the orders that do not have an invoice.
     *
     * @param {Number[]} orderIds The orders to be validated.
     *
     * @return {Promise}
     */
    function _getOrdersWithoutDocuments(orderIds) {
        return new Promise(function (resolve, reject) {
            var data = {
                do: 'OrdersOverviewAjax/GetOrdersWithoutDocuments',
                pageToken: jse.core.config.get('pageToken'),
                type: 'invoice',
                orderIds: orderIds
            };

            $.getJSON(adminRequestUrl, data).done(function (orderIdsWithoutDocument) {
                return resolve({ orderIds: orderIds, orderIdsWithoutDocument: orderIdsWithoutDocument });
            }).fail(reject);
        });
    }

    /**
     * Validate selected orders and generate/remove the orders without documents.
     *
     * @param {Object} selection Contains the "orderIds" and "orderIdsWithoutDocument" properties.
     *
     * @return {Promise} Returns the promises of the delegated methods.
     */
    function _handleMissingDocuments(selection) {
        // Indicates if missing invoices should be created
        var createMissingInvoices = $this.find(createMissingInvoicesSelector).prop('checked');

        if (createMissingInvoices) {
            return _createMissingDocuments(selection.orderIds, selection.orderIdsWithoutDocument);
        } else {
            return _removeOrderIdsWithoutDocument(selection.orderIds, selection.orderIdsWithoutDocument);
        }
    }

    /**
     * Create Missing Order Documents and set the new latest invoice id selector for which no invoice has yet existed.
     *
     * @param {Number[]} orderIds Selected order IDs.
     * @param {Number[]} orderIdsWithoutDocument Order IDs that do not have a document.
     *
     * @return {Promise} Returns a promise that will be resolved with all the order IDs.
     */
    function _createMissingDocuments(orderIds, orderIdsWithoutDocument) {
        return new Promise(function (resolve) {
            var _$;

            var requests = [];

            orderIdsWithoutDocument.forEach(function (id) {
                var url = gmPdfOrderUrl + ('?oID=' + id + '&type=invoice&ajax=1');
                var request = $.getJSON(url);

                request.done(function (response) {
                    $(emailListItemSelector).each(function (index, emailListItem) {
                        var $emailListItem = $(emailListItem);

                        if ($emailListItem.data('order').id === parseInt(id)) {
                            $emailListItem.find(latestInvoiceIdSelector).val(response.invoiceId);
                            return false;
                        }
                    });
                });

                requests.push(request);
            });

            return (_$ = $).when.apply(_$, requests).done(function () {
                return resolve(orderIds);
            });
        });
    }

    /**
     * Remove order IDs that do not have a document.
     *
     * @param {Number[]} orderIds Selected order IDs.
     * @param {Number[]} orderIdsWithoutDocument Order IDs that do not have a document.
     *
     * @return {Promise} Returns a promise that will be resolved with the orders that do have a document.
     */
    function _removeOrderIdsWithoutDocument(orderIds, orderIdsWithoutDocument) {
        return new Promise(function (resolve) {
            var orderIdsWithDocument = orderIds.filter(function (orderId) {
                return !orderIdsWithoutDocument.includes(String(orderId));
            });
            resolve(orderIdsWithDocument);
        });
    }

    /**
     * Send Invoice Emails
     *
     * @param {Number[]} orderIds Contains the IDs of the orders to be finally sent with an email.
     */
    function _sendInvoiceEmails(orderIds) {
        return new Promise(function (resolve, reject) {
            var _$2;

            var createMissingInvoices = $this.find(createMissingInvoicesSelector).prop('checked');
            var $emailListItems = $this.find(emailListItemSelector);

            // Abort and hide modal on empty email list entries.
            if (!$emailListItems.length || !orderIds.length) {
                var title = jse.core.lang.translate('TITLE_INVOICE', 'gm_order_menu');
                var message = jse.core.lang.translate('NO_RECORDS_ERROR', 'orders');
                jse.libs.modal.showMessage(title, message);
                $this.modal('hide');
                return;
            }

            // Show loading spinner.
            _toggleSpinner(true);

            // Collection of requests in promise format.
            var requests = [];

            // Fill orders array with data.
            $emailListItems.each(function (index, emailListItem) {
                var orderData = $(emailListItem).data('order');

                if (!orderIds.includes(orderData.id)) {
                    return true; // Current order does not have an invoice document.
                }

                // Email address entered in input field.
                var email = $(emailListItem).find(emailListItemEmailSelector).val();

                // The latest invoice id of the order
                var invoiceId = $(emailListItem).find(latestInvoiceIdSelector).val();

                // Request GET parameters to send.
                var parameters = {
                    oID: orderData.id,
                    type: 'invoice',
                    mail: '1',
                    bulk: '1',
                    gm_quick_mail: '1'
                };

                var url = gmPdfOrderUrl + '?' + $.param(parameters);
                var data = {
                    gm_mail: email,
                    gm_subject: module.bindings.subject.get(),
                    create_missing_invoices: Number(createMissingInvoices) // 1 or 0
                };

                if (invoiceId !== '0') {
                    data.invoice_ids = [invoiceId];
                }

                // Create AJAX request.
                requests.push($.ajax({ method: 'POST', url: url, data: data }));
            });

            (_$2 = $).when.apply(_$2, requests).done(resolve).fail(reject);
        });
    }

    /**
     * Send the invoice emails.
     *
     * This method will only send emails for the orders that do have an invoice document. If the
     * "create-missing-invoices" checkbox is active, new invoices will be generated for the orders that
     * are do not have one.
     */
    function _onSendClick() {
        var orderIds = [];

        $this.find(emailListItemSelector).each(function (index, emailListItem) {
            orderIds.push($(emailListItem).data('order').id);
        });

        _getOrdersWithoutDocuments(orderIds).then(_handleMissingDocuments).then(_sendInvoiceEmails).then(_handleDeliverySuccess).catch(_handleDeliveryFailure);
    }

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        $this.on('click', '.btn.send', _onSendClick);
        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
